from django.core.management.base import BaseCommand
from django.db import connection
from django.conf import settings
from django.core.mail import send_mail
from django.utils import timezone

class Command(BaseCommand):
    help = "One-command health check (DB + email + channels layer)."

    def add_arguments(self, parser):
        parser.add_argument("--email", action="store_true", help="Send a test email to CONSULTATION_NOTIFY_EMAIL.")
        parser.add_argument("--to", default="", help="Override recipient email for test mail.")
        parser.add_argument("--subject", default="ENABLED Tutors • Healthcheck", help="Subject for test mail.")

    def handle(self, *args, **opts):
        self.stdout.write("== ENABLED Tutors Healthcheck ==")

        # DB
        try:
            with connection.cursor() as cursor:
                cursor.execute("SELECT 1")
                row = cursor.fetchone()
            self.stdout.write(self.style.SUCCESS(f"DB: OK (engine={connection.vendor}, result={row[0] if row else 'n/a'})"))
        except Exception as e:
            self.stdout.write(self.style.ERROR(f"DB: FAIL ({e})"))

        # Channels
        try:
            layer = getattr(settings, "CHANNEL_LAYERS", None)
            backend = layer.get("default", {}).get("BACKEND") if isinstance(layer, dict) else None
            self.stdout.write(self.style.SUCCESS(f"Channels: OK ({backend or 'default'})"))
        except Exception as e:
            self.stdout.write(self.style.ERROR(f"Channels: FAIL ({e})"))

        # Email (optional)
        if opts.get("email"):
            recipient = opts.get("to") or getattr(settings, "CONSULTATION_NOTIFY_EMAIL", "") or getattr(settings, "EMAIL_HOST_USER", "")
            if not recipient:
                self.stdout.write(self.style.ERROR("Email: SKIP (no recipient; set CONSULTATION_NOTIFY_EMAIL or EMAIL_HOST_USER)"))
                return
            try:
                body = f"Healthcheck email sent at {timezone.now().isoformat()}\n\nBackend: {settings.EMAIL_BACKEND}"
                send_mail(opts.get("subject"), body, getattr(settings, "DEFAULT_FROM_EMAIL", None), [recipient], fail_silently=False)
                self.stdout.write(self.style.SUCCESS(f"Email: OK (sent to {recipient})"))
            except Exception as e:
                self.stdout.write(self.style.ERROR(f"Email: FAIL ({e})"))
