from django.conf import settings
from django.core.mail import send_mail
from django.db.models.signals import post_save
from django.dispatch import receiver

from .models import SupportMessage, SupportThread


def _safe_send_email(to_email: str, subject: str, body: str) -> None:
    """Send email safely without crashing requests/consumers if SMTP isn't configured."""
    if not to_email:
        return
    try:
        send_mail(
            subject=subject,
            message=body,
            from_email=getattr(settings, "DEFAULT_FROM_EMAIL", None),
            recipient_list=[to_email],
            fail_silently=True,  # IMPORTANT: never break chat if email fails
        )
    except Exception:
        # Extra guard (some SMTP libs can throw even with fail_silently)
        return


def _pick_staff_email(thread: SupportThread) -> str:
    """Choose who to notify when a user/guest sends a support message."""
    if thread.assigned_staff and getattr(thread.assigned_staff, "email", ""):
        return thread.assigned_staff.email

    fallback = getattr(settings, "SUPPORT_NOTIFY_STAFF_EMAIL", "")
    if fallback:
        return fallback

    # last resort: do nothing (no staff email configured)
    return ""


@receiver(post_save, sender=SupportMessage)
def support_message_email_notifications(sender, instance: SupportMessage, created: bool, **kwargs):
    if not created:
        return

    thread = instance.thread

    # Staff replied -> notify user (if authenticated user has an email)
    if instance.sender_label == "staff":
        user_email = ""
        if thread.user and getattr(thread.user, "email", ""):
            user_email = thread.user.email

        # If guest: no email stored, so we can't email them (unless you later add guest_email field)
        if user_email:
            subject = "Support reply from Enabled Tutors"
            body = (
                "Hello,\n\n"
                "You have a new reply from our support team:\n\n"
                f"{instance.text}\n\n"
                "Open the website and click ET-chat to reply.\n\n"
                "— Enabled Tutors Support"
            )
            _safe_send_email(user_email, subject, body)
        return

    # User/guest sent -> notify staff (assigned staff, or fallback staff email)
    staff_email = _pick_staff_email(thread)
    if staff_email:
        who = "Guest" if thread.user is None else f"{thread.user.get_full_name() or thread.user.username}"
        subject = f"New ET-chat message from {who}"
        body = (
            "New message received in ET-chat support inbox.\n\n"
            f"Thread ID: {thread.id}\n"
            f"From: {who}\n\n"
            f"{instance.text}\n\n"
            "Login to the admin/support inbox to reply."
        )
        _safe_send_email(staff_email, subject, body)
