from __future__ import annotations

from django.core.management.base import BaseCommand
from django.conf import settings
from django.core.mail import get_connection, EmailMessage
from django.db import connection
from asgiref.sync import async_to_sync
from channels.layers import get_channel_layer
import os


class Command(BaseCommand):
    help = "One-command health check: DB + Email + Channels. Exit code 0 if OK, 1 if any check fails."

    def add_arguments(self, parser):
        parser.add_argument("--send-email", action="store_true", help="Actually send a test email (default: only opens SMTP).")
        parser.add_argument("--email-to", default="", help="Recipient for test email (default: SUPPORT_NOTIFY_STAFF_EMAIL).")

    def handle(self, *args, **opts):
        ok = True

        # -------------------------
        # DB check
        # -------------------------
        self.stdout.write("🔎 Checking database...")
        try:
            connection.ensure_connection()
            with connection.cursor() as cur:
                cur.execute("SELECT 1;")
                cur.fetchone()
            self.stdout.write(self.style.SUCCESS(f"✅ DB OK ({connection.vendor})"))
        except Exception as e:
            ok = False
            self.stdout.write(self.style.ERROR(f"❌ DB FAIL: {e!r}"))

        # -------------------------
        # Email check (SMTP)
        # -------------------------
        self.stdout.write("🔎 Checking email (SMTP)...")
        try:
            conn = get_connection(fail_silently=False)
            conn.open()
            self.stdout.write(self.style.SUCCESS("✅ SMTP connection OK"))
            if opts.get("send_email"):
                to_email = (opts.get("email_to") or getattr(settings, "SUPPORT_NOTIFY_STAFF_EMAIL", "") or "").strip()
                if not to_email:
                    raise RuntimeError("No recipient provided. Use --email-to or set SUPPORT_NOTIFY_STAFF_EMAIL.")
                msg = EmailMessage(
                    subject="ET-chat Healthcheck Email",
                    body="Healthcheck: email sending works.",
                    from_email=getattr(settings, "DEFAULT_FROM_EMAIL", None),
                    to=[to_email],
                    connection=conn,
                )
                msg.send(fail_silently=False)
                self.stdout.write(self.style.SUCCESS(f"✅ Test email sent to {to_email}"))
            conn.close()
        except Exception as e:
            ok = False
            self.stdout.write(self.style.ERROR(f"❌ EMAIL FAIL: {e!r}"))

        # -------------------------
        # Channels check
        # -------------------------
        self.stdout.write("🔎 Checking channels layer...")
        try:
            layer = get_channel_layer()
            if layer is None:
                raise RuntimeError("No channel layer configured (CHANNEL_LAYERS is missing).")

            ch = async_to_sync(layer.new_channel)("healthcheck")
            async_to_sync(layer.send)(ch, {"type": "healthcheck.noop", "text": "ping"})
            msg = async_to_sync(layer.receive)(ch)
            if not msg:
                raise RuntimeError("No message received from channel layer.")
            self.stdout.write(self.style.SUCCESS("✅ Channels OK"))
        except Exception as e:
            ok = False
            self.stdout.write(self.style.ERROR(f"❌ CHANNELS FAIL: {e!r}"))

        if not ok:
            raise SystemExit(1)

        self.stdout.write(self.style.SUCCESS("🎉 All health checks passed."))
