from django.contrib import admin
from .models import Booking
from .matching import find_best_tutor
from chat.models import Conversation

@admin.register(Booking)
class BookingAdmin(admin.ModelAdmin):
    actions = ["auto_match_assign"]

    @admin.action(description="Auto-match tutor by subject/exam")
    def auto_match_assign(self, request, queryset):
        updated = 0
        for b in queryset:
            if b.tutor_id:
                continue
            best = find_best_tutor(b.subject, b.exam, b.service)
            if best:
                b.tutor = best
                b.status = Booking.STATUS_ASSIGNED
                b.save(update_fields=["tutor","status","updated_at"])
                Conversation.objects.get_or_create(booking=b, defaults={"student": b.student, "tutor": b.tutor})
                updated += 1
        self.message_user(request, f"Auto-matched {updated} booking(s).")

    list_display = ("id", "service", "student", "tutor", "status", "preferred_date", "created_at")
    list_filter = ("status", "service")
    search_fields = ("student__email", "student__username", "tutor__email", "service", "subject", "exam")
